// Copyright 2021-2023 by Oxford Semantic Technologies Limited.

#ifndef CRDFOX_H_
#define CRDFOX_H_

#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdbool.h>
#ifdef __cplusplus
#include <cstddef>
#include <exception>
#include <memory>
#include <optional>
#include <string>
#include <utility>
#endif

#ifdef _WIN64
    #ifdef CRDFOX_EXPORT
        #define CRDFOX __declspec(dllexport)
    #else
        #define CRDFOX __declspec(dllimport)
    #endif
#else
    #define CRDFOX
#endif

// ------------------------
// Auxiliary types
// ------------------------

typedef uint8_t CAccessTypeFlags;
typedef size_t CDataStoreParts;
typedef uint8_t CDatatypeID;
typedef uint32_t CTupleTableID;

#ifdef __cplusplus

    enum class CAccessTypeFlag : uint8_t {
        NONE  = 0x00,
        READ  = 0x01,
        WRITE = 0x02,
        GRANT = 0x04,
        FULL  = 0x80
    };

    enum class CBinaryDataStoreFormat : uint8_t {
        RAW      = 0,
        STANDARD = 1
    };

    enum class CComponentInfoPropertyType : uint8_t {
        NUMERIC = 0,
        STRING  = 1
    };

    enum class CCursorType : uint8_t {
        ASK_QUERY                   = 0,
        SELECT_QUERY                = 1,
        CONSTRUCT_NO_NAMED_GRAPHS   = 2,
        CONSTRUCT_WITH_NAMED_GRAPHS = 3
    };

    enum class CDataStorePart : size_t {
        DATA_STORE_PROPERTIES = 0x01,
        PREFIXES              = 0x02,
        FACTS                 = 0x04,
        COMMIT_PROCEDURE      = 0x08,
        DELTA_QUERIES         = 0x10,
        AXIOMS                = 0x20,
        RULES                 = 0x40,
    };

    enum class CDeltaQueryAnswerType : uint8_t {
        SNAPSHOT = 0,
        DELETION = 1,
        ADDITION = 2
    };

    enum class CDeltaQueryEvaluationMode : uint8_t {
        SNAPSHOT = 0,
        DELTA    = 1
    };

    enum class CExplanationType : uint8_t {
        SHORTEST    = 0,
        TO_EXPLICIT = 1,
        EXHAUSTIVE  = 2
    };

    enum class CInstanceState : uint8_t {
        UNINITIALIZED    = 0,
        REQUIRES_UPGRADE = 1,
        READY            = 2
    };

    enum class COfflineRequestResult : uint8_t {
        SUCCESS                          = 0,
        ALREADY_OFFLINE                  = 1,
        FAILED_DUE_TO_ACTIVE_CONNECTIONS = 2
    };

    enum class CPasswordSpecificationType : uint8_t {
        PASSWORD      = 0,
        PASSWORD_HASH = 1,
        NONE          = 2
    };

    enum class CPrefixes_DeclareResult : uint8_t {
        INVALID_PREFIX_NAME = 0,
        NO_CHANGE           = 1,
        REPLACED_EXISTING   = 2,
        DECLARED_NEW        = 3
    };

    enum class CTransactionState : uint8_t {
        READ_WRITE = 0,
        READ_ONLY  = 1,
        NONE       = 2
    };

    enum class CTransactionType : uint8_t {
        READ_WRITE = 0,
        READ_ONLY  = 1
    };

    enum class CTupleTableType : uint8_t {
        MEMORY      = 0,
        BUILTIN     = 1,
        DATA_SOURCE = 2
    };

    enum class CUpdateType : uint8_t {
        ADDITION                 = 0,
        ADDITION_UPDATE_PREFIXES = 11,
        DELETION                 = 19
    };

#else

    typedef uint8_t CAccessTypeFlag;
    const CAccessTypeFlag CAccessTypeFlag_NONE  = 0x00;
    const CAccessTypeFlag CAccessTypeFlag_READ  = 0x01;
    const CAccessTypeFlag CAccessTypeFlag_WRITE = 0x02;
    const CAccessTypeFlag CAccessTypeFlag_GRANT = 0x04;
    const CAccessTypeFlag CAccessTypeFlag_FULL  = 0x80;

    typedef uint8_t CBinaryDataStoreFormat;
    const CBinaryDataStoreFormat CBinaryDataStoreFormat_RAW      = 0;
    const CBinaryDataStoreFormat CBinaryDataStoreFormat_STANDARD = 1;

    typedef uint8_t CComponentInfoPropertyType;
    const CComponentInfoPropertyType CComponentInfoPropertyType_NUMERIC = 0;
    const CComponentInfoPropertyType CComponentInfoPropertyType_STRING  = 1;

    typedef uint8_t CCursorType;
    const CCursorType CCursorType_ASK_QUERY                   = 0;
    const CCursorType CCursorType_SELECT_QUERY                = 1;
    const CCursorType CCursorType_CONSTRUCT_NO_NAMED_GRAPHS   = 2;
    const CCursorType CCursorType_CONSTRUCT_WITH_NAMED_GRAPHS = 3;

    typedef size_t CDataStorePart;
    const CDataStorePart DataStorePart_DATA_STORE_PROPERTIES = 0x01;
    const CDataStorePart DataStorePart_PREFIXES              = 0x02;
    const CDataStorePart DataStorePart_FACTS                 = 0x04;
    const CDataStorePart DataStorePart_COMMIT_PROCEDURE      = 0x08;
    const CDataStorePart DataStorePart_DELTA_QUERIES         = 0x10;
    const CDataStorePart DataStorePart_AXIOMS                = 0x20;
    const CDataStorePart DataStorePart_RULES                 = 0x40;

    typedef uint8_t CDeltaQueryAnswerType;
    const CDeltaQueryAnswerType CDeltaQueryAnswerType_SNAPSHOT = 0;
    const CDeltaQueryAnswerType CDeltaQueryAnswerType_DELETION = 1;
    const CDeltaQueryAnswerType CDeltaQueryAnswerType_ADDITION = 2;

    typedef uint8_t CDeltaQueryEvaluationMode;
    const CDeltaQueryEvaluationMode CDeltaQueryEvaluationMode_SNAPSHOT = 0;
    const CDeltaQueryEvaluationMode CDeltaQueryEvaluationMode_DELTA    = 1;

    typedef uint8_t CExplanationType;
    const CExplanationType CExplanationType_SHORTEST    = 0;
    const CExplanationType CExplanationType_TO_EXPLICIT = 1;
    const CExplanationType CExplanationType_EXHAUSTIVE  = 2;

    typedef uint8_t CInstanceState;
    const CInstanceState CInstanceState_UNINITIALIZED    = 0;
    const CInstanceState CInstanceState_REQUIRES_UPGRADE = 1;
    const CInstanceState CInstanceState_READY            = 2;

    typedef uint8_t COfflineRequestResult;
    const COfflineRequestResult COfflineRequestResult_SUCCESS                          = 0;
    const COfflineRequestResult COfflineRequestResult_ALREADY_OFFLINE                  = 1;
    const COfflineRequestResult COfflineRequestResult_FAILED_DUE_TO_ACTIVE_CONNECTIONS = 2;

    typedef uint8_t CPasswordSpecificationType;
    const CPasswordSpecificationType CPasswordSpecificationType_PASSWORD      = 0;
    const CPasswordSpecificationType CPasswordSpecificationType_PASSWORD_HASH = 1;
    const CPasswordSpecificationType CPasswordSpecificationType_NONE          = 2;

    typedef uint8_t CPrefixes_DeclareResult;
    const CPrefixes_DeclareResult CPrefixes_DeclareResult_INVALID_PREFIX_NAME = 0;
    const CPrefixes_DeclareResult CPrefixes_DeclareResult_NO_CHANGE           = 1;
    const CPrefixes_DeclareResult CPrefixes_DeclareResult_REPLACED_EXISTING   = 2;
    const CPrefixes_DeclareResult CPrefixes_DeclareResult_DECLARED_NEW        = 3;

    typedef uint8_t CTransactionState;
    const CTransactionState CTransactionState_READ_WRITE = 0;
    const CTransactionState CTransactionState_READ_ONLY  = 1;
    const CTransactionState CTransactionState_NONE       = 2;

    typedef uint8_t CTransactionType;
    const CTransactionType CTransactionType_READ_WRITE = 0;
    const CTransactionType CTransactionType_READ_ONLY  = 1;

    typedef uint8_t CTupleTableType;
    const CTupleTableType CTupleTableType_MEMORY      = 0;
    const CTupleTableType CTupleTableType_BUILTIN     = 1;
    const CTupleTableType CTupleTableType_DATA_SOURCE = 2;

    typedef uint8_t CUpdateType;
    const CUpdateType CUpdateType_ADDITION                 = 0;
    const CUpdateType CUpdateType_ADDITION_UPDATE_PREFIXES = 11;
    const CUpdateType CUpdateType_DELETION                 = 19;

#endif

const int64_t LOCK_TIMEOUT_WAIT_FOREVER = -1;

typedef const char* (*CDataStoreOperationStartMonitor)(void* context, size_t dataStoreVersionOnTransactionStart);

typedef struct CInputStream {
    void* context;
    bool (*rewindFn)(void* context);
    bool (*readFn)(void* context, void* data, size_t numberOfBytesToRead, size_t* bytesRead);
} CInputStream;

typedef struct COutputStream {
    void* context;
    bool (*flushFn)(void* context);
    bool (*writeFn)(void* context, const void* data, size_t numberOfBytesToWrite);
} COutputStream;

typedef struct {
    bool statementTypeSupportsUserQuery;
    size_t numberOfQueryAnswers;
    size_t totalNumberOfQueryAnswers;
    bool statementTypeSupportsDeletions;
    size_t numberOfAttemptedDeletions;
    size_t numberOfDeletions;
    bool statementTypeSupportsInsertions;
    size_t numberOfAttemptedInsertions;
    size_t numberOfInsertions;
} CStatementResult;

// ------------------------
// Forward declaration of CRDFox interface objects
// ------------------------

#ifdef __cplusplus
    struct CCommitProcedureInfo;
    struct CComponentInfo;
    struct CCursor;
    struct CDataSourceInfo;
    struct CDataSourceInfoMap;
    struct CDataSourceTableInfo;
    struct CDataSourceTableInfoList;
    struct CDataStoreConnection;
    struct CDataStoreInfo;
    struct CDataStoreInfoMap;
    struct CDataStorePropertiesMap;
    struct CDeltaQueryAnswerInfo;
    struct CDeltaQueryAnswerInfoList;
    struct CDeltaQueryInfo;
    struct CDeltaQueryInfoMap;
    struct CParameters;
    struct CPrefixes;
    struct CPrivilegesMap;
    struct CResourceValueTuples;
    struct CServerConnection;
    struct CSetOfStrings;
    struct CStatisticsInfo;
    struct CStatisticsInfoList;
    struct CTupleTableInfo;
    struct CTupleTableInfoMap;
#else
    typedef struct { unsigned char __dummy; } CCommitProcedureInfo;
    typedef struct { unsigned char __dummy; } CComponentInfo;
    typedef struct { unsigned char __dummy; } CCursor;
    typedef struct { unsigned char __dummy; } CDataSourceInfo;
    typedef struct { unsigned char __dummy; } CDataSourceInfoMap;
    typedef struct { unsigned char __dummy; } CDataSourceTableInfo;
    typedef struct { unsigned char __dummy; } CDataSourceTableInfoList;
    typedef struct { unsigned char __dummy; } CDataStoreConnection;
    typedef struct { unsigned char __dummy; } CDataStoreInfo;
    typedef struct { unsigned char __dummy; } CDataStoreInfoMap;
    typedef struct { unsigned char __dummy; } CDataStoreTableInfo;
    typedef struct { unsigned char __dummy; } CDataStoreTableInfoList;
    typedef struct { unsigned char __dummy; } CDataStorePropertiesMap;
    typedef struct { unsigned char __dummy; } CDeltaQueryAnswerInfo;
    typedef struct { unsigned char __dummy; } CDeltaQueryAnswerInfoList;
    typedef struct { unsigned char __dummy; } CDeltaQueryInfo;
    typedef struct { unsigned char __dummy; } CDeltaQueryInfoMap;
    typedef struct { unsigned char __dummy; } CParameters;
    typedef struct { unsigned char __dummy; } CPrefixes;
    typedef struct { unsigned char __dummy; } CPrivilegesMap;
    typedef struct { unsigned char __dummy; } CResourceValueTuples;
    typedef struct { unsigned char __dummy; } CServerConnection;
    typedef struct { unsigned char __dummy; } CSetOfStrings;
    typedef struct { unsigned char __dummy; } CStatisticsInfo;
    typedef struct { unsigned char __dummy; } CStatisticsInfoList;
    typedef struct { unsigned char __dummy; } CTupleTableInfo;
    typedef struct { unsigned char __dummy; } CTupleTableInfoMap;
#endif

// ------------------------
// CRDFox interface functions
// ------------------------

#ifdef __cplusplus
extern "C" {
    #define CRDFOX_NOEXCEPT noexcept
#else
    #define CRDFOX_NOEXCEPT
#endif

extern const COutputStream* const RDFOX_STDOUT;

extern const char* const GUESS_FORMAT_NAME;

// ------------------------
// CCommitProcedureInfo
// ------------------------

CRDFOX void CCommitProcedureInfo_destroy(CCommitProcedureInfo* commitProcedureInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CCommitProcedureInfo_getText(const CCommitProcedureInfo* commitProcedureInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CComponentInfo
// ------------------------

CRDFOX void CComponentInfo_destroy(CComponentInfo* componentInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CComponentInfo_getName(const CComponentInfo* componentInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CComponentInfo_getNumberOfProperties(const CComponentInfo* componentInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CComponentInfo_getPropertyName(const CComponentInfo* componentInfo, size_t propertyIndex) CRDFOX_NOEXCEPT;

CRDFOX CComponentInfoPropertyType CComponentInfo_getPropertyTypeByIndex(const CComponentInfo* componentInfo, size_t propertyIndex) CRDFOX_NOEXCEPT;

CRDFOX bool CComponentInfo_getPropertyTypeByName(const CComponentInfo* componentInfo, const char* propertyName, CComponentInfoPropertyType* propertyType) CRDFOX_NOEXCEPT;

CRDFOX bool CComponentInfo_getNumericPropertyValueByIndex(const CComponentInfo* componentInfo, size_t propertyIndex, size_t* integerValue, uint16_t* thousandthsValue) CRDFOX_NOEXCEPT;

CRDFOX bool CComponentInfo_getNumericPropertyValueByName(const CComponentInfo* componentInfo, const char* propertyName, size_t* integerValue, uint16_t* thousandthsValue) CRDFOX_NOEXCEPT;

CRDFOX bool CComponentInfo_getStringPropertyValueByIndex(const CComponentInfo* componentInfo, size_t propertyIndex, const char** stringValue) CRDFOX_NOEXCEPT;

CRDFOX bool CComponentInfo_getStringPropertyValueByName(const CComponentInfo* componentInfo, const char* propertyName, const char** stringValue) CRDFOX_NOEXCEPT;

CRDFOX size_t CComponentInfo_getNumberOfSubComponents(const CComponentInfo* componentInfo) CRDFOX_NOEXCEPT;

CRDFOX const CComponentInfo* CComponentInfo_getSubComponent(const CComponentInfo* componentInfo, size_t subcomponentInfoIndex) CRDFOX_NOEXCEPT;

// ------------------------
// CCursor
// ------------------------

CRDFOX void CCursor_destroy(CCursor* cursor) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getDataStoreConnection(CCursor* cursor, CDataStoreConnection** dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getType(CCursor* cursor, CCursorType* cursorType) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getArity(CCursor* cursor, size_t* arity) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getAnswerVariableName(CCursor* cursor, size_t variableIndex, const char** answerVariableName) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_open(CCursor* cursor, size_t skipToOffset, size_t* multiplicity) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_canAdvance(CCursor* cursor, bool* canAdvance) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_advance(CCursor* cursor, size_t* multiplicity) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getCurrentOffset(CCursor* cursor, size_t* currentOffset) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getCurrentMultiplicity(CCursor* cursor, size_t* currentMultiplicity) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_produceQueryAnswers(CCursor* cursor, const COutputStream* outputStream, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce, size_t* numberOfProducedAnswers) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_produceQueryAnswersToBuffer(CCursor* cursor, char* buffer, size_t bufferSize, size_t* resultSize, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce, size_t* numberOfProducedAnswers) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_produceQueryAnswersToFile(CCursor* cursor, const char* filePath, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce, size_t* numberOfProducedAnswers) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_stop(CCursor* cursor) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_getDatatypeID(CCursor* cursor, size_t variableIndex, CDatatypeID* datatypeID) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_appendResourceLexicalForm(CCursor* cursor, size_t variableIndex, char* buffer, size_t bufferSize, size_t* lexicalFormSize, CDatatypeID* datatypeID, bool* resourceResolved) CRDFOX_NOEXCEPT;

CRDFOX bool CCursor_appendResourceTurtleLiteral(CCursor* cursor, size_t variableIndex, char* buffer, size_t bufferSize, size_t* turtleLiteralSize, bool* resourceResolved) CRDFOX_NOEXCEPT;

// ------------------------
// CDataSourceInfo
// ------------------------

CRDFOX void CDataSourceInfo_destroy(CDataSourceInfo* dataSourceInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataSourceInfo_getName(const CDataSourceInfo* dataSourceInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataSourceInfo_getNumberOfTables(const CDataSourceInfo* dataSourceInfo) CRDFOX_NOEXCEPT;

CRDFOX const CParameters* CDataSourceInfo_getParameters(const CDataSourceInfo* dataSourceInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CDataSourceInfoMap
// ------------------------

CRDFOX void CDataSourceInfoMap_destroy(CDataSourceInfoMap* dataSourceInfoMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataSourceInfoMap_size(const CDataSourceInfoMap* dataSourceInfoMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataSourceInfoMap_getKey(const CDataSourceInfoMap* dataSourceInfoMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX const CDataSourceInfo* CDataSourceInfoMap_getValue(const CDataSourceInfoMap* dataSourceInfoMap, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CDataSourceTableInfo
// ------------------------

CRDFOX void CDataSourceTableInfo_destroy(CDataSourceTableInfo* dataSourceTableInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataSourceTableInfo_getName(const CDataSourceTableInfo* dataSourceTableInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataSourceTableInfo_getNumberOfColumns(const CDataSourceTableInfo* dataSourceTableInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataSourceTableInfo_getColumnName(const CDataSourceTableInfo* dataSourceTableInfo, size_t columnIndex) CRDFOX_NOEXCEPT;

CRDFOX CDatatypeID CDataSourceTableInfo_getColumnDatatypeID(const CDataSourceTableInfo* dataSourceTableInfo, size_t columnIndex) CRDFOX_NOEXCEPT;

// ------------------------
// CDataSourceTableInfoList
// ------------------------

CRDFOX void CDataSourceTableInfoList_destroy(CDataSourceTableInfoList* dataSourceTableInfoList) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataSourceTableInfoList_size(const CDataSourceTableInfoList* dataSourceTableInfoList) CRDFOX_NOEXCEPT;

CRDFOX const CDataSourceTableInfo* CDataSourceTableInfoList_get(const CDataSourceTableInfoList* dataSourceTableInfoList, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CDataStoreConnection
// ------------------------

CRDFOX void CDataStoreConnection_destroy(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

// Connection management

CRDFOX bool CDataStoreConnection_interrupt(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getAgentName(CDataStoreConnection* dataStoreConnection, const char** agentName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getLockTimeout(CDataStoreConnection* dataStoreConnection, int64_t* lockTimeoutMS) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setLockTimeout(CDataStoreConnection* dataStoreConnection, int64_t lockTimeoutMS) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getDataStoreOperationStartMonitor(CDataStoreConnection* dataStoreConnection, void** context, CDataStoreOperationStartMonitor* dataStoreOperationStartMonitor) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setDataStoreOperationStartMonitor(CDataStoreConnection* dataStoreConnection, void* context, CDataStoreOperationStartMonitor dataStoreOperationStartMonitor) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_duplicate(CDataStoreConnection* dataStoreConnection, CDataStoreConnection** outputDataStoreConnection) CRDFOX_NOEXCEPT;

// Data store information

CRDFOX bool CDataStoreConnection_getName(CDataStoreConnection* dataStoreConnection, const char** name) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getUniqueID(CDataStoreConnection* dataStoreConnection, const char** uniqueID) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_isPersistent(CDataStoreConnection* dataStoreConnection, bool* isPersistent) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getParameters(CDataStoreConnection* dataStoreConnection, const CParameters** parameters) CRDFOX_NOEXCEPT;

// Detailed statistics

CRDFOX bool CDataStoreConnection_getComponentInfo(CDataStoreConnection* dataStoreConnection, bool extended, CComponentInfo** componentInfo) CRDFOX_NOEXCEPT;

// Data store properties

CRDFOX bool CDataStoreConnection_getDataStoreProperties(CDataStoreConnection* dataStoreConnection, CDataStorePropertiesMap** dataStorePropertiesMap) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getDataStoreProperty(CDataStoreConnection* dataStoreConnection, const char* propertyName, char* buffer, size_t bufferSize, size_t* propertyValueLength) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setDataStoreProperty(CDataStoreConnection* dataStoreConnection, const char* propertyName, const char* propertyValue, bool* changed) CRDFOX_NOEXCEPT;

// Prefixes management

CRDFOX bool CDataStoreConnection_getPrefixes(CDataStoreConnection* dataStoreConnection, CPrefixes** prefixes) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setPrefixes(CDataStoreConnection* dataStoreConnection, const CPrefixes* prefixes, bool* changed) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setPrefix(CDataStoreConnection* dataStoreConnection, const char* prefixName, const char* prefixIRI, bool* changed) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_unsetPrefix(CDataStoreConnection* dataStoreConnection, const char* prefixName, bool* changed) CRDFOX_NOEXCEPT;

// Data source management

CRDFOX bool CDataStoreConnection_listDataSources(CDataStoreConnection* dataStoreConnection, CDataSourceInfoMap** dataSourceInfoMap) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_containsDataSource(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_describeDataSource(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, CDataSourceInfo** dataSourceInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_registerDataSource(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, const CParameters* dataSourceParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_deregisterDataSource(CDataStoreConnection* dataStoreConnection, const char* dataSourceName) CRDFOX_NOEXCEPT;

// Data source table management

CRDFOX bool CDataStoreConnection_listDataSourceTables(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, CDataSourceTableInfoList** dataSourceTableInfoList) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_containsDataSourceTable(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, const char* dataSourceTableName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_describeDataSourceTable(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, const char* dataSourceTableName, CDataSourceTableInfo** dataSourceTableInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getDataSourceTableData(CDataStoreConnection* dataStoreConnection, const char* dataSourceName, const char* dataSourceTableName, size_t numberOfRows, CResourceValueTuples** resourceValueTuples) CRDFOX_NOEXCEPT;

// Tuple table management

CRDFOX bool CDataStoreConnection_listTupleTables(CDataStoreConnection* dataStoreConnection, CTupleTableInfoMap** tupleTableInfoMap) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_containsTupleTable(CDataStoreConnection* dataStoreConnection, const char* tupleTableName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_describeTupleTable(CDataStoreConnection* dataStoreConnection, const char* tupleTableName, CTupleTableInfo** tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_createTupleTable(CDataStoreConnection* dataStoreConnection, const char* tupleTableName, const CParameters* tupleTableParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_deleteTupleTable(CDataStoreConnection* dataStoreConnection, const char* tupleTableName) CRDFOX_NOEXCEPT;

// Statistics management

CRDFOX bool CDataStoreConnection_listStatistics(CDataStoreConnection* dataStoreConnection, CStatisticsInfoList** statisticsInfoList) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_containsStatistics(CDataStoreConnection* dataStoreConnection, const char* statisticsName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_describeStatistics(CDataStoreConnection* dataStoreConnection, const char* statisticsName, CStatisticsInfo** statisticsInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_createStatistics(CDataStoreConnection* dataStoreConnection, const char* statisticsName, const CParameters* statisticsParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_deleteStatistics(CDataStoreConnection* dataStoreConnection, const char* statisticseName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_updateStatistics(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_updateStatisticsWithName(CDataStoreConnection* dataStoreConnection, const char* statisticseName) CRDFOX_NOEXCEPT;

// Commit procedure management

CRDFOX bool CDataStoreConnection_getCommitProcedure(CDataStoreConnection* dataStoreConnection, CCommitProcedureInfo** commitProcedureInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setCommitProcedure(CDataStoreConnection* dataStoreConnection, const char* commitProcedureText, size_t commitProcedureTextLength, bool* changed) CRDFOX_NOEXCEPT;

// Delta query management

CRDFOX bool CDataStoreConnection_listDeltaQueries(CDataStoreConnection* dataStoreConnection, CDeltaQueryInfoMap** deltaQueryInfoMap) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_containsDeltaQuery(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_describeDeltaQuery(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, CDeltaQueryInfo** deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_registerDeltaQuery(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, const char* queryText, size_t queryTextLength, const char* formatName, size_t snapshotsMaxSize, size_t deletionsMaxSize, size_t additionsMaxSize) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_deregisterDeltaQuery(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_setDeltaQueryEvaluationMode(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, CDeltaQueryEvaluationMode deltaQueryEvaluationMode) CRDFOX_NOEXCEPT;

// Delta query answer management

CRDFOX bool CDataStoreConnection_listDeltaQueryAnswers(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, const size_t* fromDataStoreVersion, const size_t* toDataStoreVersion, const CDeltaQueryAnswerType* deltaQueryAnswerType, CDeltaQueryAnswerInfoList** deltaQueryAnswerInfoList) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_deleteDeltaQueryAnswers(CDataStoreConnection* dataStoreConnection, const char* deltaQueryName, const size_t* fromDataStoreVersion, const size_t* toDataStoreVersion, const CDeltaQueryAnswerType* deltaQueryAnswerType, size_t* numberOfDeletedDeltaQueryAnswers) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_exportDeltaQueryAnswer(CDataStoreConnection* dataStoreConnection, const COutputStream* outputStream, const char* deltaQueryName, size_t dataStoreVersion, const CDeltaQueryAnswerType deltaQueryAnswerType) CRDFOX_NOEXCEPT;

// Transaction management

CRDFOX bool CDataStoreConnection_getDataStoreVersionAfterLastOperation(CDataStoreConnection* dataStoreConnection, size_t* dataStoreVersion) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getDataStoreVersion(CDataStoreConnection* dataStoreConnection, size_t* dataStoreVersion) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getLastSnapshotDataStoreVersion(CDataStoreConnection* dataStoreConnection, size_t* lastSnapshotDataStoreVersion) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_getTransactionState(CDataStoreConnection* dataStoreConnection, CTransactionState* transactionState) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_transactionRequiresRollback(CDataStoreConnection* dataStoreConnection, bool* transactionRequiresRollback) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_beginTransaction(CDataStoreConnection* dataStoreConnection, CTransactionType transactionType) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_commitTransaction(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_rollbackTransaction(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

// Various operations on the data store

CRDFOX bool CDataStoreConnection_compact(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_clear(CDataStoreConnection* dataStoreConnection, CDataStoreParts dataStoreParts) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_clearRulesAxiomsExplicateFacts(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

// Data import/export

CRDFOX bool CDataStoreConnection_exportData(CDataStoreConnection* dataStoreConnection, const COutputStream* outputStream, const char* formatName, const CParameters* parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_exportDataToBuffer(CDataStoreConnection* dataStoreConnection, char* buffer, size_t bufferSize, size_t* resultSize, const char* formatName, const CParameters* parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_exportDataToFile(CDataStoreConnection* dataStoreConnection, const char* filePath, const char* formatName, const CParameters* parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_importData(CDataStoreConnection* dataStoreConnection, const char* defaultGraphName, CUpdateType updateType, const CInputStream* inputStream, const char* baseIRI, const char* formatName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_importDataFromBuffer(CDataStoreConnection* dataStoreConnection, const char* defaultGraphName, CUpdateType updateType, const char* buffer, size_t bufferLength, const char* formatName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_importDataFromFile(CDataStoreConnection* dataStoreConnection, const char* defaultGraphName, CUpdateType updateType, const char* filePath, const char* formatName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_importDataFromURI(CDataStoreConnection* dataStoreConnection, const char* defaultGraphName, CUpdateType updateType, const char* uri, const char* formatName) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_importAxiomsFromTriples(CDataStoreConnection* dataStoreConnection, const char* sourceGraphName, bool translateAssertions, const char* destinationGraphName, CUpdateType updateType) CRDFOX_NOEXCEPT;

// Management of the axioms

// Management of the rules

CRDFOX bool CDataStoreConnection_recompile(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_printRulePlans(CDataStoreConnection* dataStoreConnection, const COutputStream* outputStream) CRDFOX_NOEXCEPT;

// Management of the materialization

CRDFOX bool CDataStoreConnection_requiresIncrementalReasoning(CDataStoreConnection* dataStoreConnection, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_updateMaterialization(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_recomputeMaterialization(CDataStoreConnection* dataStoreConnection) CRDFOX_NOEXCEPT;

// Explanation

CRDFOX bool CDataStoreConnection_createExplanation(CDataStoreConnection* dataStoreConnection, const char* factText, size_t factTextLength, size_t maxDistanceFromRoot, size_t maxRuleInstancesPerFact, CExplanationType explanationType, const COutputStream* outputStream) CRDFOX_NOEXCEPT;

// Query & update evaluation

CRDFOX bool CDataStoreConnection_createCursor(CDataStoreConnection* dataStoreConnection, const char* queryText, size_t queryTextLength, const CParameters* compilationParameters, CCursor** cursor) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateQuery(CDataStoreConnection* dataStoreConnection, const char* queryText, size_t queryTextLength, const CParameters* compilationParameters, const COutputStream* outputStream, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateQueryToBuffer(CDataStoreConnection* dataStoreConnection, const char* queryText, size_t queryTextLength, const CParameters* compilationParameters, char* buffer, size_t bufferSize, size_t* resultSize, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateQueryToFile(CDataStoreConnection* dataStoreConnection, const char* queryText, size_t queryTextLength, const CParameters* compilationParameters, const char* filePath, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateUpdate(CDataStoreConnection* dataStoreConnection, const char* updateText, size_t updateTextLength, const CParameters* compilationParameters, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateStatement(CDataStoreConnection* dataStoreConnection, const char* statementText, size_t statementTextLength, const CParameters* compilationParameters, const COutputStream* outputStream, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateStatementToBuffer(CDataStoreConnection* dataStoreConnection, const char* statementText, size_t statementTextLength, const CParameters* compilationParameters, char* buffer, size_t bufferSize, size_t* resultSize, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreConnection_evaluateStatementToFile(CDataStoreConnection* dataStoreConnection, const char* statementText, size_t statementTextLength, const CParameters* compilationParameters, const char* filePath, const char* queryAnswerFormatName, CStatementResult* statementResult) CRDFOX_NOEXCEPT;

// ------------------------
// CDataStoreInfo
// ------------------------

CRDFOX void CDataStoreInfo_destroy(CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataStoreInfo_getName(const CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataStoreInfo_getUniqueID(const CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreInfo_isPersistent(const CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CDataStoreInfo_isOnline(const CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX const CParameters* CDataStoreInfo_getParameters(const CDataStoreInfo* dataStoreInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CDataStoreInfoMap
// ------------------------

CRDFOX void CDataStoreInfoMap_destroy(CDataStoreInfoMap* dataStoreInfoMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataStoreInfoMap_size(const CDataStoreInfoMap* dataStoreInfoMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataStoreInfoMap_getKey(const CDataStoreInfoMap* dataStoreInfoMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX const CDataStoreInfo* CDataStoreInfoMap_getValue(const CDataStoreInfoMap* dataStoreInfoMap, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CDataStorePropertiesMap
// ------------------------

CRDFOX void CDataStorePropertiesMap_destroy(CDataStorePropertiesMap* dataStorePropertiesMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CDataStorePropertiesMap_size(const CDataStorePropertiesMap* dataStorePropertiesMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataStorePropertiesMap_getKey(const CDataStorePropertiesMap* dataStorePropertiesMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX const char* CDataStorePropertiesMap_getValue(const CDataStorePropertiesMap* dataStorePropertiesMap, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CDeltaQueryAnswerInfo
// ------------------------

CRDFOX void CDeltaQueryAnswerInfo_destroy(CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryAnswerInfo_getDeltaQueryName(const CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryAnswerInfo_getDataStoreVersion(const CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

CRDFOX CDeltaQueryAnswerType CDeltaQueryAnswerInfo_getDeltaQueryAnswerType(const CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryAnswerInfo_getPath(const CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryAnswerInfo_getFormatName(const CDeltaQueryAnswerInfo* deltaQueryAnswerInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CDeltaQueryAnswerInfoList
// ------------------------

CRDFOX void CDeltaQueryAnswerInfoList_destroy(CDeltaQueryAnswerInfoList* deltaQueryAnswerInfoList) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryAnswerInfoList_size(const CDeltaQueryAnswerInfoList* deltaQueryAnswerInfoList) CRDFOX_NOEXCEPT;

CRDFOX const CDeltaQueryAnswerInfo* CDeltaQueryAnswerInfoList_get(const CDeltaQueryAnswerInfoList* deltaQueryAnswerInfoList, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CDeltaQueryInfo
// ------------------------

CRDFOX void CDeltaQueryInfo_destroy(CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryInfo_getName(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryInfo_getUniqueID(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryInfo_getQueryText(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryInfo_getFormatName(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getSnapshotsMaxSize(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getDeletionsMaxSize(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getAdditionsMaxSize(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getLastDataStoreVersionWithSnapshot(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getLastDataStoreVersionWithDeletion(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfo_getLastDataStoreVersionWithAddition(const CDeltaQueryInfo* deltaQueryInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CDeltaQueryInfoMap
// ------------------------

CRDFOX void CDeltaQueryInfoMap_destroy(CDeltaQueryInfoMap* tupleTableInfoMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CDeltaQueryInfoMap_size(const CDeltaQueryInfoMap* tupleTableInfoMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CDeltaQueryInfoMap_getKey(const CDeltaQueryInfoMap* tupleTableInfoMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX const CDeltaQueryInfo* CDeltaQueryInfoMap_getValue(const CDeltaQueryInfoMap* tupleTableInfoMap, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CParameters
// ------------------------

CRDFOX const CParameters* CParameters_getEmptyParameters(void) CRDFOX_NOEXCEPT;

CRDFOX void CParameters_destroy(CParameters* parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CParameters_newEmptyParameters(CParameters** parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CParameters_setString(CParameters* parameters, const char* key, const char* value) CRDFOX_NOEXCEPT;

CRDFOX bool CParameters_getString(const CParameters* parameters, const char* key, const char* const defaultValue, const char** string) CRDFOX_NOEXCEPT;

// ------------------------
// CPrefixes
// ------------------------

CRDFOX const CPrefixes* CPrefixes_getEmptyPrefixes(void) CRDFOX_NOEXCEPT;

CRDFOX const CPrefixes* CPrefixes_getDefaultPrefixes(void) CRDFOX_NOEXCEPT;

CRDFOX void CPrefixes_destroy(CPrefixes* prefixes) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_newEmptyPrefixes(CPrefixes** prefixes) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_newDefaultPrefixes(CPrefixes** prefixes) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_declareStandardPrefixes(CPrefixes* prefixes) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_declarePrefix(CPrefixes* prefixes, const char* prefixName, const char* prefixIRI, CPrefixes_DeclareResult* declareResult) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_undeclarePrefix(CPrefixes* prefixes, const char* prefixName, bool* changed) CRDFOX_NOEXCEPT;

CRDFOX bool CPrefixes_getPrefix(const CPrefixes* prefixes, const char* prefixName, const char** prefixIRI) CRDFOX_NOEXCEPT;

// ------------------------
// CPrivilegesMap
// ------------------------

CRDFOX void CPrivilegesMap_destroy(CPrivilegesMap* privilegesMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CPrivilegesMap_size(const CPrivilegesMap* privilegesMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CPrivilegesMap_getResourceSpecifier(const CPrivilegesMap* privilegesMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX CAccessTypeFlags CPrivilegesMap_getGrantedAccessTypes(const CPrivilegesMap* privilegesMap, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CRDFoxServer
// ------------------------

CRDFOX void CRDFoxServer_getLastException(const char** what, size_t* whatLength, bool* isRDFoxException, const char** exceptionName, size_t* exceptionNameLength, const char** message, size_t* messageLength) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_analyzeState(const char* serverDirectoryPath, CInstanceState* instanceState) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_initialize(const char* serverDirectoryPath, const char* persistenceEncryptionKey, const CParameters* persistentServerParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_upgrade(const char* serverDirectoryPath, const char* persistenceEncryptionKey, const COutputStream* outputStream) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_start(const CParameters* serverParameters, size_t* numberOfDataStoresInServer) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_getNumberOfRoles(size_t* numberOfRoles) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_createFirstRole(const char* firstRoleName, const char* password) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_newServerConnection(const char* roleName, const char* password, CServerConnection** serverConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_newDataStoreConnection(const char* dataStoreName, const char* roleName, const char* password, CDataStoreConnection** dataStoreConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CRDFoxServer_stop(void) CRDFOX_NOEXCEPT;

// ------------------------
// CResourceValueTuples
// ------------------------

CRDFOX void CResourceValueTuples_destroy(CResourceValueTuples* resourceValueTuples) CRDFOX_NOEXCEPT;

CRDFOX size_t CResourceValueTuples_size(const CResourceValueTuples* resourceValueTuples) CRDFOX_NOEXCEPT;

CRDFOX size_t CResourceValueTuples_arity(const CResourceValueTuples* resourceValueTuples) CRDFOX_NOEXCEPT;

CRDFOX bool CResourceValueTuples_appendResourceLexicalForm(const CResourceValueTuples* resourceValueTuples, size_t row, size_t column, char* buffer, size_t bufferSize, size_t* lexicalFormSize, CDatatypeID* datatypeID) CRDFOX_NOEXCEPT;

CRDFOX bool CResourceValueTuples_appendResourceTurtleLiteral(const CResourceValueTuples* resourceValueTuples, const CPrefixes* prefixes, size_t row, size_t column, char* buffer, size_t bufferSize, size_t* turtleLiteralSize) CRDFOX_NOEXCEPT;

// ------------------------
// CServerConnection
// ------------------------

CRDFOX void CServerConnection_destroy(CServerConnection* serverConnection) CRDFOX_NOEXCEPT;

// Connection management

CRDFOX bool CServerConnection_interrupt(CServerConnection* serverConnection) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_duplicate(CServerConnection* serverConnection, CServerConnection** outputServerConnection) CRDFOX_NOEXCEPT;

// Server properties

CRDFOX bool CServerConnection_getParameters(CServerConnection* serverConnection, const CParameters** parameters) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getRDFoxVersion(CServerConnection* serverConnection, const char** rdfoxVersion) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getRDFoxGitSHA(CServerConnection* serverConnection, const char** rdfoxGitSHA) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getMemoryUse(CServerConnection* serverConnection, size_t* maxUsedBytes, size_t* availableBytes) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_setMaxMemoryUse(CServerConnection* serverConnection, size_t maxUsedBytes) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getNumberOfThreads(CServerConnection* serverConnection, size_t* numberOfThreads) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_setNumberOfThreads(CServerConnection* serverConnection, size_t numberOfThreads) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getServerVersion(CServerConnection* serverConnection, size_t* serverVersion) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_checkIntegrity(CServerConnection* serverConnection, size_t maxErrorsOfSameType, const COutputStream* outputStream, bool* errorsDetected) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getComponentInfo(CServerConnection* serverConnection, bool extended, CComponentInfo** componentInfo) CRDFOX_NOEXCEPT;

// Data store management

CRDFOX bool CServerConnection_listDataStores(CServerConnection* serverConnection, CDataStoreInfoMap** dataStoreInfoMap) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_containsDataStore(CServerConnection* serverConnection, const char* dataStoreName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_describeDataStore(CServerConnection* serverConnection, const char* dataStoreName, CDataStoreInfo** dataStoreInfo) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_createDataStore(CServerConnection* serverConnection, const char* dataStoreName, const CParameters* dataStoreParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_loadDataStoreFromBinaryFormat(CServerConnection* serverConnection, const char* dataStoreName, const CInputStream* inputStream) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_loadDataStoreFromBinaryFormatWithParameters(CServerConnection* serverConnection, const char* dataStoreName, const CInputStream* inputStream, const CParameters* dataStoreParameters) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_saveDataStoreToBinaryFormat(CServerConnection* serverConnection, const char* dataStoreName, const COutputStream* outputStream, CBinaryDataStoreFormat format) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_saveDataStoreToBinaryFormatWithMonitor(CServerConnection* serverConnection, const char* dataStoreName, const COutputStream* outputStream, CBinaryDataStoreFormat format, void* context, CDataStoreOperationStartMonitor dataStoreOperationStartMonitor) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_deleteDataStore(CServerConnection* serverConnection, const char* dataStoreName) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_isDataStoreOnline(CServerConnection* serverConnection, const char* dataStoreName, bool* isOnline) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_bringDataStoreOnline(CServerConnection* serverConnection, const char* dataStoreName, bool* wasOffline) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_bringDataStoreOffline(CServerConnection* serverConnection, const char* dataStoreName, COfflineRequestResult* offlineRequestResult) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_newDataStoreConnection(CServerConnection* serverConnection, const char* dataStoreName, CDataStoreConnection** dataStoreConnection) CRDFOX_NOEXCEPT;

// Role management

CRDFOX bool CServerConnection_listRoles(CServerConnection* serverConnection, CSetOfStrings** setOfRoles) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_createRole(CServerConnection* serverConnection, const char* roleName, CPasswordSpecificationType passwordSpecificationType, const char* password) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_deleteRole(CServerConnection* serverConnection, const char* roleName) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_changeRolePassword(CServerConnection* serverConnection, const char* currentPassword, const char* newPassword) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_getRolePasswordHash(CServerConnection* serverConnection, const char* roleName, char* buffer, size_t bufferSize, size_t* rolePasswordHashLength) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_listRoleMemberships(CServerConnection* serverConnection, const char* roleName, CSetOfStrings** setOfRoleMemberships) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_listRoleMembers(CServerConnection* serverConnection, const char* roleName, CSetOfStrings** setOfRoleMembers) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_isMemberOf(CServerConnection* serverConnection, const char* roleName, const char* superRoleName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_grantRole(CServerConnection* serverConnection, const char* roleName, const char* superRoleName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_revokeRole(CServerConnection* serverConnection, const char* roleName, const char* superRoleName, bool* result) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_listPrivileges(CServerConnection* serverConnection, const char* roleName, CPrivilegesMap** privilegesMap) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_grantPrivileges(CServerConnection* serverConnection, const char* roleName, const char* resourceSpecifier, CAccessTypeFlags accessTypesToGrant, bool* granted) CRDFOX_NOEXCEPT;

CRDFOX bool CServerConnection_revokePrivileges(CServerConnection* serverConnection, const char* roleName, const char* resourceSpecifier, CAccessTypeFlags accessTypesToRevoke, bool* revoked) CRDFOX_NOEXCEPT;

// ------------------------
// CSetOfStrings
// ------------------------

CRDFOX void CSetOfStrings_destroy(CSetOfStrings* setOfStrings) CRDFOX_NOEXCEPT;

CRDFOX size_t CSetOfStrings_size(const CSetOfStrings* setOfStrings) CRDFOX_NOEXCEPT;

CRDFOX const char* CSetOfStrings_get(const CSetOfStrings* setOfStrings, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CStatisticsInfo
// ------------------------

CRDFOX void CStatisticsInfo_destroy(CStatisticsInfo* statisticsInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CStatisticsInfo_getName(const CStatisticsInfo* statisticsInfo) CRDFOX_NOEXCEPT;

CRDFOX const CParameters* CStatisticsInfo_getParameters(const CStatisticsInfo* statisticsInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CStatisticsInfoList
// ------------------------

CRDFOX void CStatisticsInfoList_destroy(CStatisticsInfoList* statisticsInfoList) CRDFOX_NOEXCEPT;

CRDFOX size_t CStatisticsInfoList_size(const CStatisticsInfoList* statisticsInfoList) CRDFOX_NOEXCEPT;

CRDFOX const CStatisticsInfo* CStatisticsInfoList_get(const CStatisticsInfoList* statisticsInfoList, size_t index) CRDFOX_NOEXCEPT;

// ------------------------
// CTupleTableInfo
// ------------------------

CRDFOX void CTupleTableInfo_destroy(CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX const char* CTupleTableInfo_getName(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX CTupleTableType CTupleTableInfo_getType(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX CTupleTableID CTupleTableInfo_getID(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CTupleTableInfo_getMinArity(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX size_t CTupleTableInfo_getMaxArity(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

CRDFOX const CParameters* CTupleTableInfo_getParameters(const CTupleTableInfo* tupleTableInfo) CRDFOX_NOEXCEPT;

// ------------------------
// CTupleTableInfoMap
// ------------------------

CRDFOX void CTupleTableInfoMap_destroy(CTupleTableInfoMap* tupleTableInfoMap) CRDFOX_NOEXCEPT;

CRDFOX size_t CTupleTableInfoMap_size(const CTupleTableInfoMap* tupleTableInfoMap) CRDFOX_NOEXCEPT;

CRDFOX const char* CTupleTableInfoMap_getKey(const CTupleTableInfoMap* tupleTableInfoMap, size_t index) CRDFOX_NOEXCEPT;

CRDFOX const CTupleTableInfo* CTupleTableInfoMap_getValue(const CTupleTableInfoMap* tupleTableInfoMap, size_t index) CRDFOX_NOEXCEPT;

#undef CRDFOX_NOEXCEPT
#ifdef __cplusplus
}

// ------------------------
// Declarations of smart pointers to CRDFox interface objects
// ------------------------

#define DECLARE_OBJECTPTR(type)                            \
    struct type##Deleter {                                 \
        inline void operator()(type* object) noexcept {    \
            ::type##_destroy(object);                      \
        }                                                  \
    };                                                     \
    typedef std::unique_ptr<type, type##Deleter> type##Ptr

DECLARE_OBJECTPTR(CCommitProcedureInfo);
DECLARE_OBJECTPTR(CComponentInfo);
DECLARE_OBJECTPTR(CCursor);
DECLARE_OBJECTPTR(CDataSourceInfo);
DECLARE_OBJECTPTR(CDataSourceInfoMap);
DECLARE_OBJECTPTR(CDataSourceTableInfo);
DECLARE_OBJECTPTR(CDataSourceTableInfoList);
DECLARE_OBJECTPTR(CDataStoreConnection);
DECLARE_OBJECTPTR(CDataStoreInfo);
DECLARE_OBJECTPTR(CDataStoreInfoMap);
DECLARE_OBJECTPTR(CDataStorePropertiesMap);
DECLARE_OBJECTPTR(CDeltaQueryAnswerInfo);
DECLARE_OBJECTPTR(CDeltaQueryAnswerInfoList);
DECLARE_OBJECTPTR(CDeltaQueryInfo);
DECLARE_OBJECTPTR(CDeltaQueryInfoMap);
DECLARE_OBJECTPTR(CParameters);
DECLARE_OBJECTPTR(CPrefixes);
DECLARE_OBJECTPTR(CPrivilegesMap);
DECLARE_OBJECTPTR(CResourceValueTuples);
DECLARE_OBJECTPTR(CServerConnection);
DECLARE_OBJECTPTR(CSetOfStrings);
DECLARE_OBJECTPTR(CStatisticsInfo);
DECLARE_OBJECTPTR(CStatisticsInfoList);
DECLARE_OBJECTPTR(CTupleTableInfo);
DECLARE_OBJECTPTR(CTupleTableInfoMap);

#undef DECLARE_OBJECTPTR

// ------------------------
// Hourglass pattern for C++
// ------------------------

class CRDFoxException : public std::exception {

protected:

    std::string m_what;
    bool m_isRDFoxException;
    std::string m_exceptionName;
    std::string m_message;

    inline explicit CRDFoxException(std::string&& what, bool isRDFoxException, std::string&& exceptionName, std::string&& message) noexcept :
        m_what(std::move(what)),
        m_isRDFoxException(isRDFoxException),
        m_exceptionName(std::move(exceptionName)),
        m_message(std::move(message))
    {
    }

public:

    [[noreturn]] static void reportLastException() {
        const char* what;
        size_t whatLength;
        bool isRDFoxException;
        const char* exceptionName;
        size_t exceptionNameLength;
        const char* message;
        size_t messageLength;
        ::CRDFoxServer_getLastException(&what, &whatLength, &isRDFoxException, &exceptionName, &exceptionNameLength, &message, &messageLength);
        throw CRDFoxException(std::string{ what, whatLength }, isRDFoxException, std::string{ exceptionName, exceptionNameLength }, std::string{ message, messageLength });
    }

    const char* what() const noexcept override {
        return m_what.c_str();
    }

    inline bool isRDFoxException() const noexcept {
        return m_isRDFoxException;
    }

    inline const char* getExceptionName() const noexcept {
        return m_exceptionName.c_str();
    }

    inline const char* getMessage() const noexcept {
        return m_message.c_str();
    }

};

#define REPORT_EXCEPTION_IF_NEEDED(expression) \
    if (expression)                            \
        CRDFoxException::reportLastException()

#define DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(type) \
    type() = delete;                             \
    type(const type&) = delete;                  \
    type(type&&) = delete;                       \
    type& operator=(const type&) = delete;       \
    type& operator=(type&&) = delete

// ------------------------
// CCommitProcedureInfo
// ------------------------

struct CCommitProcedureInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CCommitProcedureInfo);

    inline const char* getText() const noexcept {
        return ::CCommitProcedureInfo_getText(this);
    }

};

// ------------------------
// CComponentInfo
// ------------------------

struct CComponentInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CComponentInfo);

    inline const char* getName() const noexcept {
        return ::CComponentInfo_getName(this);
    }

    inline size_t getNumberOfProperties() const noexcept {
        return ::CComponentInfo_getNumberOfProperties(this);
    }

    inline const char* getPropertyName(size_t propertyIndex) const noexcept {
        return ::CComponentInfo_getPropertyName(this, propertyIndex);
    }

    inline CComponentInfoPropertyType getPropertyType(size_t propertyIndex) const noexcept {
        return ::CComponentInfo_getPropertyTypeByIndex(this, propertyIndex);
    }

    inline CComponentInfoPropertyType getPropertyType(const char* propertyName) const {
        CComponentInfoPropertyType propertyType;
        REPORT_EXCEPTION_IF_NEEDED(::CComponentInfo_getPropertyTypeByName(this, propertyName, &propertyType));
        return propertyType;
    }

    inline std::pair<size_t, uint16_t> getNumericPropertyValue(size_t propertyIndex) const {
        size_t integerValue;
        uint16_t thousandthsValue;
        REPORT_EXCEPTION_IF_NEEDED(::CComponentInfo_getNumericPropertyValueByIndex(this, propertyIndex, &integerValue, &thousandthsValue));
        return std::make_pair(integerValue, thousandthsValue);
    }

    inline std::pair<size_t, uint16_t> getNumericPropertyValue(const char* propertyName) const {
        size_t integerValue;
        uint16_t thousandthsValue;
        REPORT_EXCEPTION_IF_NEEDED(::CComponentInfo_getNumericPropertyValueByName(this, propertyName, &integerValue, &thousandthsValue));
        return std::make_pair(integerValue, thousandthsValue);
    }

    inline const char* getStringPropertyValue(size_t propertyIndex) const {
        const char* stringValue;
        REPORT_EXCEPTION_IF_NEEDED(::CComponentInfo_getStringPropertyValueByIndex(this, propertyIndex, &stringValue));
        return stringValue;
    }

    inline const char* getStringPropertyValue(const char* propertyName) const {
        const char* stringValue;
        REPORT_EXCEPTION_IF_NEEDED(::CComponentInfo_getStringPropertyValueByName(this, propertyName, &stringValue));
        return stringValue;
    }

    inline size_t getNumberOfSubComponents() const noexcept {
        return ::CComponentInfo_getNumberOfSubComponents(this);
    }

    inline const CComponentInfo& getSubComponent(size_t subcomponentInfoIndex) const noexcept {
        return *::CComponentInfo_getSubComponent(this, subcomponentInfoIndex);
    }

};

// ------------------------
// CCursor
// ------------------------

struct CCursor {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CCursor);

    inline CDataStoreConnection& getDataStoreConnection() {
        CDataStoreConnection* dataStoreConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getDataStoreConnection(this, &dataStoreConnection));
        return *dataStoreConnection;
    }

    inline CCursorType getType() {
        CCursorType cursorType;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getType(this, &cursorType));
        return cursorType;
    }

    inline size_t getArity() {
        size_t arity;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getArity(this, &arity));
        return arity;
    }

    inline const char* getAnswerVariableName(size_t variableIndex) {
        const char* answerVariableName;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getAnswerVariableName(this, variableIndex, &answerVariableName));
        return answerVariableName;
    }

    inline size_t open(size_t skipToOffset) {
        size_t multiplicity;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_open(this, skipToOffset, &multiplicity));
        return multiplicity;
    }

    inline bool canAdvance() {
        bool canAdvance;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_canAdvance(this, &canAdvance));
        return canAdvance;
    }

    inline size_t advance() {
        size_t multiplicity;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_advance(this, &multiplicity));
        return multiplicity;
    }

    inline size_t getCurrentOffset() {
        size_t currentOffset;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getCurrentOffset(this, &currentOffset));
        return currentOffset;
    }

    inline size_t getCurrentMultiplicity() {
        size_t currentMultiplicity;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getCurrentMultiplicity(this, &currentMultiplicity));
        return currentMultiplicity;
    }

    inline size_t produceQueryAnswers(const COutputStream& outputStream, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce) {
        size_t numberOfProducedAnswers;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_produceQueryAnswers(this, &outputStream, queryAnswerFormatName, maxNumberOfAnswersToProduce, &numberOfProducedAnswers));
        return numberOfProducedAnswers;
    }

    inline size_t produceQueryAnswersToBuffer(char* buffer, size_t bufferSize, size_t& resultSize, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce) {
        size_t numberOfProducedAnswers;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_produceQueryAnswersToBuffer(this, buffer, bufferSize, &resultSize, queryAnswerFormatName, maxNumberOfAnswersToProduce, &numberOfProducedAnswers));
        return numberOfProducedAnswers;
    }

    inline size_t produceQueryAnswersToFile(const char* filePath, const char* queryAnswerFormatName, size_t maxNumberOfAnswersToProduce) {
        size_t numberOfProducedAnswers;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_produceQueryAnswersToFile(this, filePath, queryAnswerFormatName, maxNumberOfAnswersToProduce, &numberOfProducedAnswers));
        return numberOfProducedAnswers;
    }

    inline void stop() {
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_stop(this));
    }

    inline CDatatypeID getDatatypeID(size_t variableIndex) {
        CDatatypeID datatypeID;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_getDatatypeID(this, variableIndex, &datatypeID));
        return datatypeID;
    }

    inline bool appendResourceLexicalForm(size_t variableIndex, char* buffer, size_t bufferSize, size_t& lexicalFormSize, CDatatypeID& datatypeID) {
        bool resourceResolved;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_appendResourceLexicalForm(this, variableIndex, buffer, bufferSize, &lexicalFormSize, &datatypeID, &resourceResolved));
        return resourceResolved;
    }

    inline bool appendResourceTurtleLiteral(size_t variableIndex, char* buffer, size_t bufferSize, size_t& turtleLiteralSize) {
        bool resourceResolved;
        REPORT_EXCEPTION_IF_NEEDED(::CCursor_appendResourceTurtleLiteral(this, variableIndex, buffer, bufferSize, &turtleLiteralSize, &resourceResolved));
        return resourceResolved;
    }

};

// ------------------------
// CDataSourceInfo
// ------------------------

struct CDataSourceInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataSourceInfo);

    inline const char* getName() const noexcept {
        return ::CDataSourceInfo_getName(this);
    }

    inline size_t getNumberOfTables() const noexcept {
        return ::CDataSourceInfo_getNumberOfTables(this);
    }

    inline const CParameters& getParameters() const noexcept {
        return *::CDataSourceInfo_getParameters(this);
    }

};

// ------------------------
// CDataSourceInfoMap
// ------------------------

struct CDataSourceInfoMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataSourceInfoMap);

    inline size_t size() const noexcept {
        return ::CDataSourceInfoMap_size(this);
    }

    inline const char* getKey(size_t index) const noexcept {
        return ::CDataSourceInfoMap_getKey(this, index);
    }

    inline bool hasValue(size_t index) const noexcept {
        return ::CDataSourceInfoMap_getValue(this, index) != nullptr;
    }

    inline const CDataSourceInfo& getValue(size_t index) const noexcept {
        return *::CDataSourceInfoMap_getValue(this, index);
    }

};

// ------------------------
// CDataSourceTableInfo
// ------------------------

struct CDataSourceTableInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataSourceTableInfo);

    inline const char* getName() const noexcept {
        return ::CDataSourceTableInfo_getName(this);
    }

    inline size_t getNumberOfColumns() const noexcept {
        return ::CDataSourceTableInfo_getNumberOfColumns(this);
    }

    inline const char* getColumnName(size_t columnIndex) const noexcept {
        return ::CDataSourceTableInfo_getColumnName(this, columnIndex);
    }

    inline CDatatypeID getColumnDatatypeID(size_t columnIndex) const noexcept {
        return ::CDataSourceTableInfo_getColumnDatatypeID(this, columnIndex);
    }

};

// ------------------------
// CDataSourceTableInfoList
// ------------------------
struct CDataSourceTableInfoList {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataSourceTableInfoList);

    inline size_t size() const noexcept {
        return ::CDataSourceTableInfoList_size(this);
    }

    inline const CDataSourceTableInfo& get(size_t index) const noexcept {
        return *::CDataSourceTableInfoList_get(this, index);
    }
};

// ------------------------
// CDataStoreConnection
// ------------------------

struct CDataStoreConnection {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataStoreConnection);

    // Connection management

    inline void interrupt() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_interrupt(this));
    }

    inline const char* getAgentName() {
        const char* agentName;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getAgentName(this, &agentName));
        return agentName;
    }

    inline int64_t getLockTimeout() {
        int64_t lockTimeoutMS;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getLockTimeout(this, &lockTimeoutMS));
        return lockTimeoutMS;
    }

    inline void setLockTimeout(int64_t lockTimeoutMS) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setLockTimeout(this, lockTimeoutMS));
    }

    inline void getDataStoreOperationStartMonitor(void* & context, CDataStoreOperationStartMonitor& dataStoreOperationStartMonitor) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataStoreOperationStartMonitor(this, &context, &dataStoreOperationStartMonitor));
    }

    inline void setDataStoreOperationStartMonitor(void* context, CDataStoreOperationStartMonitor dataStoreOperationStartMonitor) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setDataStoreOperationStartMonitor(this, context, dataStoreOperationStartMonitor));
    }

    inline CDataStoreConnectionPtr duplicate() {
        CDataStoreConnection* outputDataStoreConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_duplicate(this, &outputDataStoreConnection));
        return CDataStoreConnectionPtr{ outputDataStoreConnection };
    }

    // Data store information

    inline const char* getName() {
        const char* name;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getName(this, &name));
        return name;
    }

    inline const char* getUniqueID() {
        const char* uniqueID;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getUniqueID(this, &uniqueID));
        return uniqueID;
    }

    inline bool isPersistent() {
        bool isPersistent;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_isPersistent(this, &isPersistent));
        return isPersistent;
    }

    inline const CParameters& getParameters() {
        const CParameters* parameters;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getParameters(this, &parameters));
        return *parameters;
    }

    // Detailed statistics

    CComponentInfoPtr getComponentInfo(bool extended) {
        CComponentInfo* componentInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getComponentInfo(this, extended, &componentInfo));
        return CComponentInfoPtr{ componentInfo };
    }

    // Data store properties

    inline CDataStorePropertiesMapPtr getDataStoreProperties() {
        CDataStorePropertiesMap* dataStorePropertiesMap;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataStoreProperties(this, &dataStorePropertiesMap));
        return CDataStorePropertiesMapPtr{ dataStorePropertiesMap };
    }

    inline void getDataStoreProperty(const char* propertyName, char* buffer, size_t bufferSize, size_t* propertyValueLength) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataStoreProperty(this, propertyName, buffer, bufferSize, propertyValueLength));
    }

    inline bool setDataStoreProperty(const char* propertyName, const char* propertyValue) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setDataStoreProperty(this, propertyName, propertyValue, &changed));
        return changed;
    }

    // Prefixes management

    inline CPrefixesPtr getPrefixes() {
        CPrefixes* prefixes;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getPrefixes(this, &prefixes));
        return CPrefixesPtr{ prefixes };
    }

    inline bool setPrefixes(const CPrefixes* prefixes) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setPrefixes(this, prefixes, &changed));
        return changed;
    }

    inline bool setPrefix(const char* prefixName, const char* prefixIRI) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setPrefix(this, prefixName, prefixIRI, &changed));
        return changed;
    }

    inline bool unsetPrefix(const char* prefixName) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_unsetPrefix(this, prefixName, &changed));
        return changed;
    }

    // Data source management

    inline CDataSourceInfoMapPtr listDataSources() {
        CDataSourceInfoMap* dataSourceInfoMap;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listDataSources(this, &dataSourceInfoMap));
        return CDataSourceInfoMapPtr{ dataSourceInfoMap };
    };

    inline bool containsDataSource(const char* dataSourceName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_containsDataSource(this, dataSourceName, &result));
        return result;
    }

    inline CDataSourceInfoPtr describeDataSource(const char* dataSourceName) {
        CDataSourceInfo* dataSourceInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_describeDataSource(this, dataSourceName, &dataSourceInfo));
        return CDataSourceInfoPtr{ dataSourceInfo };
    }

    inline void registerDataSource(const char* dataSourceName, const CParameters& dataSourceParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_registerDataSource(this, dataSourceName, &dataSourceParameters));
    }

    inline void deregisterDataSource(const char* dataSourceName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_deregisterDataSource(this, dataSourceName));
    }

    // Data source table management

    inline CDataSourceTableInfoListPtr listDataSourceTables(const char* dataSourceName) {
        CDataSourceTableInfoList* dataSourceTableInfoList;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listDataSourceTables(this, dataSourceName, &dataSourceTableInfoList));
        return CDataSourceTableInfoListPtr{ dataSourceTableInfoList };
    }

    inline bool containsDataSourceTable(const char* dataSourceName, const char* dataSourceTableName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_containsDataSourceTable(this, dataSourceName, dataSourceTableName, &result));
        return result;
    }

    inline CDataSourceTableInfoPtr describeDataSourceTable(const char* dataSourceName, const char* dataSourceTableName) {
        CDataSourceTableInfo* dataSourceTableInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_describeDataSourceTable(this, dataSourceName, dataSourceTableName, &dataSourceTableInfo));
        return CDataSourceTableInfoPtr{ dataSourceTableInfo };
    }

    inline CResourceValueTuplesPtr getDataSourceTableData(const char* dataSourceName, const char* dataSourceTableName, size_t numberOfRows) {
        CResourceValueTuples* resourceValueTuples;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataSourceTableData(this, dataSourceName, dataSourceTableName, numberOfRows, &resourceValueTuples));
        return CResourceValueTuplesPtr{ resourceValueTuples };
    }

    // Tuple table management

    inline CTupleTableInfoMapPtr listTupleTables() {
        CTupleTableInfoMap* tupleTableInfoMap;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listTupleTables(this, &tupleTableInfoMap));
        return CTupleTableInfoMapPtr{ tupleTableInfoMap };
    }

    inline bool containsTupleTable(const char* tupleTableName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_containsTupleTable(this, tupleTableName, &result));
        return result;
    }

    inline CTupleTableInfoPtr describeTupleTable(const char* tupleTableName) {
        CTupleTableInfo* tupleTableInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_describeTupleTable(this, tupleTableName, &tupleTableInfo));
        return CTupleTableInfoPtr{ tupleTableInfo };
    }

    inline void createTupleTable(const char* tupleTableName, const CParameters& tupleTableParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_createTupleTable(this, tupleTableName, &tupleTableParameters));
    }

    inline void deleteTupleTable(const char* tupleTableName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_deleteTupleTable(this, tupleTableName));
    }

    // Statistics management

    inline CStatisticsInfoListPtr listStatistics() {
        CStatisticsInfoList *statisticsInfoList;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listStatistics(this, &statisticsInfoList));
        return CStatisticsInfoListPtr{ statisticsInfoList };
    };

    inline bool containsStatistics(const char* statisticsName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_containsStatistics(this, statisticsName, &result));
        return result;
    }

    inline CStatisticsInfoPtr describeStatistics(const char* statisticsName) {
        CStatisticsInfo* statisticsInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_describeStatistics(this, statisticsName, &statisticsInfo));
        return CStatisticsInfoPtr{ statisticsInfo };
    }

    inline void createStatistics(const char* statisticsName, const CParameters& statisticsParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_createStatistics(this, statisticsName, &statisticsParameters));
    }

    inline void deleteStatistics(const char* statisticsName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_deleteStatistics(this, statisticsName));
    }

    inline void updateStatistics() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_updateStatistics(this));
    }

    inline void updateStatisticsWithName(const char* statisticsName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_updateStatisticsWithName(this, statisticsName));
    }

    // Commit procedure management

    inline CCommitProcedureInfoPtr getCommitProcedure() {
        CCommitProcedureInfo* commitProcedureInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getCommitProcedure(this, &commitProcedureInfo));
        return CCommitProcedureInfoPtr{ commitProcedureInfo };
    }

    inline bool setCommitProcedure(const char* commitProcedureText, size_t commitProcedureTextLength) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setCommitProcedure(this, commitProcedureText, commitProcedureTextLength, &changed));
        return changed;
    }

    // Delta query management

    inline CDeltaQueryInfoMapPtr listDeltaQueries() {
        CDeltaQueryInfoMap* deltaQueryInfoMap;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listDeltaQueries(this, &deltaQueryInfoMap));
        return CDeltaQueryInfoMapPtr{ deltaQueryInfoMap };
    }

    inline bool containsDeltaQuery(const char* deltaQueryName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_containsDeltaQuery(this, deltaQueryName, &result));
        return result;
    }

    inline CDeltaQueryInfoPtr describeDeltaQuery(const char* deltaQueryName) {
        CDeltaQueryInfo* deltaQueryInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_describeDeltaQuery(this, deltaQueryName, &deltaQueryInfo));
        return CDeltaQueryInfoPtr{ deltaQueryInfo };
    }

    inline void registerDeltaQuery(const char* deltaQueryName, const char* queryText, size_t queryTextLength, const char* formatName, size_t snapshotsMaxSize, size_t deletionsMaxSize, size_t additionsMaxSize) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_registerDeltaQuery(this, deltaQueryName, queryText, queryTextLength, formatName, snapshotsMaxSize, deletionsMaxSize, additionsMaxSize));
    }

    inline void deregisterDeltaQuery(const char* deltaQueryName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_deregisterDeltaQuery(this, deltaQueryName));
    }

    inline void setDeltaQueryEvaluationMode(const char* deltaQueryName, CDeltaQueryEvaluationMode deltaQueryEvaluationMode) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_setDeltaQueryEvaluationMode(this, deltaQueryName, deltaQueryEvaluationMode));
    }

    // Delta query answer management

    inline CDeltaQueryAnswerInfoListPtr listDeltaQueryAnswers(const char* deltaQueryName, std::optional<size_t> fromDataStoreVersion, std::optional<size_t> toDataStoreVersion, std::optional<CDeltaQueryAnswerType> deltaQueryAnswerType) {
        CDeltaQueryAnswerInfoList* deltaQueryAnswerInfoList;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_listDeltaQueryAnswers(this, deltaQueryName, fromDataStoreVersion.has_value() ? &*fromDataStoreVersion : nullptr, toDataStoreVersion.has_value() ? &*toDataStoreVersion : nullptr, deltaQueryAnswerType.has_value() ? &*deltaQueryAnswerType : nullptr, &deltaQueryAnswerInfoList));
        return CDeltaQueryAnswerInfoListPtr{ deltaQueryAnswerInfoList };
    }

    inline size_t deleteDeltaQueryAnswers(const char* deltaQueryName, std::optional<size_t> fromDataStoreVersion, std::optional<size_t> toDataStoreVersion, std::optional<CDeltaQueryAnswerType> deltaQueryAnswerType) {
        size_t result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_deleteDeltaQueryAnswers(this, deltaQueryName, fromDataStoreVersion.has_value() ? &*fromDataStoreVersion : nullptr, toDataStoreVersion.has_value() ? &*toDataStoreVersion : nullptr, deltaQueryAnswerType.has_value() ? &*deltaQueryAnswerType : nullptr, &result));
        return result;
    }

    inline void exportDeltaQueryAnswer(const COutputStream& outputStream, const char* deltaQueryName, size_t dataStoreVersion, CDeltaQueryAnswerType deltaQueryAnswerType) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_exportDeltaQueryAnswer(this, &outputStream, deltaQueryName, dataStoreVersion, deltaQueryAnswerType));
    }

    // Transaction management

    inline size_t getDataStoreVersionAfterLastOperation() {
        size_t dataStoreVersion;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataStoreVersionAfterLastOperation(this, &dataStoreVersion));
        return dataStoreVersion;
    }

    inline size_t getDataStoreVersion() {
        size_t dataStoreVersion;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getDataStoreVersion(this, &dataStoreVersion));
        return dataStoreVersion;
    }

    inline size_t getLastSnapshotDataStoreVersion() {
        size_t lastSnapshotDataStoreVersion;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getLastSnapshotDataStoreVersion(this, &lastSnapshotDataStoreVersion));
        return lastSnapshotDataStoreVersion;
    }

    inline CTransactionState getTransactionState() {
        CTransactionState transactionState;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_getTransactionState(this, &transactionState));
        return transactionState;
    }

    inline bool transactionRequiresRollback() {
        bool transactionRequiresRollback;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_transactionRequiresRollback(this, &transactionRequiresRollback));
        return transactionRequiresRollback;
    }

    inline void beginTransaction(CTransactionType transactionType) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_beginTransaction(this, transactionType));
    }

    inline void commitTransaction() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_commitTransaction(this));
    }

    inline void rollbackTransaction() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_rollbackTransaction(this));
    }

    // Various operations on the data store

    inline void compact() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_compact(this));
    }

    inline void clear(CDataStoreParts dataStoreParts) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_clear(this, dataStoreParts));
    }

    inline void clearRulesAxiomsExplicateFacts() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_clearRulesAxiomsExplicateFacts(this));
    }

    // Data import/export

    inline void exportData(const COutputStream& outputStream, const char* formatName, const CParameters& parameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_exportData(this, &outputStream, formatName, &parameters));
    }

    inline void exportDataToBuffer(char* buffer, size_t bufferSize, size_t& resultSize, const char* formatName, const CParameters& parameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_exportDataToBuffer(this, buffer, bufferSize, &resultSize, formatName, &parameters));
    }

    inline void exportDataToFile(const char* filePath, const char* formatName, const CParameters& parameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_exportDataToFile(this, filePath, formatName, &parameters));
    }

    inline void importData(const char* defaultGraphName, CUpdateType updateType, const CInputStream& inputStream, const char* baseIRI, const char* formatName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_importData(this, defaultGraphName, updateType, &inputStream, baseIRI, formatName));
    }

    inline void importDataFromBuffer(const char* defaultGraphName, CUpdateType updateType, const char* buffer, size_t bufferLength, const char* formatName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_importDataFromBuffer(this, defaultGraphName, updateType, buffer, bufferLength, formatName));
    }

    inline void importDataFromFile(const char* defaultGraphName, CUpdateType updateType, const char* filePath, const char* formatName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_importDataFromFile(this, defaultGraphName, updateType, filePath, formatName));
    }

    inline void importDataFromURI(const char* defaultGraphName, CUpdateType updateType, const char* uri, const char* formatName) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_importDataFromURI(this, defaultGraphName, updateType, uri, formatName));
    }

    inline void importAxiomsFromTriples(const char* sourceGraphName, bool translateAssertions, const char* destinationGraphName, CUpdateType updateType) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_importAxiomsFromTriples(this, sourceGraphName, translateAssertions, destinationGraphName, updateType));
    }

    // Management of the axioms

    // Management of the rules

    inline void recompile() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_recompile(this));
    }

    inline void printRulePlans(const COutputStream& outputStream) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_printRulePlans(this, &outputStream));
    }

    // Management of the materialization

    inline bool requiresIncrementalReasoning() {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_requiresIncrementalReasoning(this, &result));
        return result;
    }

    inline void updateMaterialization() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_updateMaterialization(this));
    }

    inline void recomputeMaterialization() {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_recomputeMaterialization(this));
    }

    // Explanation

    inline void createExplanation(const char* factText, size_t factTextLength, size_t maxDistanceFromRoot, size_t maxRuleInstancesPerFact, CExplanationType explanationType, const COutputStream& outputStream) {
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_createExplanation(this, factText, factTextLength, maxDistanceFromRoot, maxRuleInstancesPerFact, explanationType, &outputStream));
    }

    // Query & update evaluation

    inline CCursorPtr createCursor(const char* queryText, size_t queryTextLength, const CParameters& compilationParameters) {
        CCursor* cursor;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_createCursor(this, queryText, queryTextLength, &compilationParameters, &cursor));
        return CCursorPtr{ cursor };
    }

    inline CStatementResult evaluateQuery(const char* queryText, size_t queryTextLength, const CParameters& compilationParameters, const COutputStream& outputStream, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateQuery(this, queryText, queryTextLength, &compilationParameters, &outputStream, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateQueryToBuffer(const char* queryText, size_t queryTextLength, const CParameters& compilationParameters, char* buffer, size_t bufferSize, size_t& resultSize, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateQueryToBuffer(this, queryText, queryTextLength, &compilationParameters, buffer, bufferSize, &resultSize, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateQueryToFile(const char* queryText, size_t queryTextLength, const CParameters& compilationParameters, const char* filePath, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateQueryToFile(this, queryText, queryTextLength, &compilationParameters, filePath, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateUpdate(const char* updateText, size_t updateTextLength, const CParameters& compilationParameters) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateUpdate(this, updateText, updateTextLength, &compilationParameters, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateStatement(const char* statementText, size_t statementTextLength, const CParameters& compilationParameters, const COutputStream& outputStream, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateStatement(this, statementText, statementTextLength, &compilationParameters, &outputStream, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateStatementToBuffer(const char* statementText, size_t statementTextLength, const CParameters& compilationParameters, char* buffer, size_t bufferSize, size_t& resultSize, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateStatementToBuffer(this, statementText, statementTextLength, &compilationParameters, buffer, bufferSize, &resultSize, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

    inline CStatementResult evaluateStatementToFile(const char* statementText, size_t statementTextLength, const CParameters& compilationParameters, const char* filePath, const char* queryAnswerFormatName) {
        CStatementResult statementResult;
        REPORT_EXCEPTION_IF_NEEDED(::CDataStoreConnection_evaluateStatementToFile(this, statementText, statementTextLength, &compilationParameters, filePath, queryAnswerFormatName, &statementResult));
        return statementResult;
    }

};

// ------------------------
// CDataStoreInfo
// ------------------------

struct CDataStoreInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataStoreInfo);

    inline const char* getName() const noexcept {
        return ::CDataStoreInfo_getName(this);
    }

    inline const char* getUniqueID() const noexcept {
        return ::CDataStoreInfo_getUniqueID(this);
    }

    inline bool isPersistent() const noexcept {
        return ::CDataStoreInfo_isPersistent(this);
    }

    inline bool isOnline() const noexcept {
        return ::CDataStoreInfo_isOnline(this);
    }

    inline const CParameters& getParameters() const noexcept {
        return *::CDataStoreInfo_getParameters(this);
    }

};

// ------------------------
// CDataStoreInfoMap
// ------------------------

struct CDataStoreInfoMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataStoreInfoMap);

    inline size_t size() const noexcept {
        return ::CDataStoreInfoMap_size(this);
    }

    inline const char* getKey(size_t index) const noexcept {
        return ::CDataStoreInfoMap_getKey(this, index);
    }

    inline const CDataStoreInfo& getValue(size_t index) const noexcept {
        return *::CDataStoreInfoMap_getValue(this, index);
    }

};

// ------------------------
// CDataStorePropertiesMap
// ------------------------

struct CDataStorePropertiesMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDataStorePropertiesMap);

    inline size_t size() const noexcept {
        return ::CDataStorePropertiesMap_size(this);
    }

    inline const char* getKey(size_t index) const noexcept {
        return ::CDataStorePropertiesMap_getKey(this, index);
    }

    inline const char* getValue(size_t index) const noexcept {
        return ::CDataStorePropertiesMap_getValue(this, index);
    }

};

// ------------------------
// CDeltaQueryAnswerInfo
// ------------------------

struct CDeltaQueryAnswerInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDeltaQueryAnswerInfo);

    inline const char* getDeltaQueryName() const noexcept {
        return ::CDeltaQueryAnswerInfo_getDeltaQueryName(this);
    }

    inline size_t getDataStoreVersion() const noexcept {
        return ::CDeltaQueryAnswerInfo_getDataStoreVersion(this);
    }

    inline CDeltaQueryAnswerType getDeltaQueryAnswerType() const noexcept {
        return ::CDeltaQueryAnswerInfo_getDeltaQueryAnswerType(this);
    }

    inline const char* getPath() const noexcept {
        return ::CDeltaQueryAnswerInfo_getPath(this);
    }

    inline const char* getFormatName() const noexcept {
        return ::CDeltaQueryAnswerInfo_getFormatName(this);
    }

};

// ------------------------
// CDeltaQueryAnswerInfoList
// ------------------------

struct CDeltaQueryAnswerInfoList {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDeltaQueryAnswerInfoList);

    inline size_t size() const noexcept {
        return ::CDeltaQueryAnswerInfoList_size(this);
    }

    inline const CDeltaQueryAnswerInfo& get(size_t index) const noexcept {
        return *::CDeltaQueryAnswerInfoList_get(this, index);
    }

};

// ------------------------
// CDeltaQueryInfo
// ------------------------

struct CDeltaQueryInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDeltaQueryInfo);

    inline const char* getName() const noexcept {
        return ::CDeltaQueryInfo_getName(this);
    }

    inline const char* getUniqueID() const noexcept {
        return ::CDeltaQueryInfo_getUniqueID(this);
    }

    inline const char* getQueryText() const noexcept {
        return ::CDeltaQueryInfo_getQueryText(this);
    }

    inline const char* getFormatName() const noexcept {
        return ::CDeltaQueryInfo_getFormatName(this);
    }

    inline size_t getSnapshotsMaxSize() const noexcept {
        return ::CDeltaQueryInfo_getSnapshotsMaxSize(this);
    }

    inline size_t getDeletionsMaxSize() const noexcept {
        return ::CDeltaQueryInfo_getDeletionsMaxSize(this);
    }

    inline size_t getAdditionsMaxSize() const noexcept {
        return ::CDeltaQueryInfo_getAdditionsMaxSize(this);
    }

    inline size_t getLastDataStoreVersionWithSnapshot() const noexcept {
        return ::CDeltaQueryInfo_getLastDataStoreVersionWithSnapshot(this);
    }

    inline size_t getLastDataStoreVersionWithDeletion() const noexcept {
        return ::CDeltaQueryInfo_getLastDataStoreVersionWithDeletion(this);
    }

    inline size_t getLastDataStoreVersionWithAddition() const noexcept {
        return ::CDeltaQueryInfo_getLastDataStoreVersionWithAddition(this);
    }

};

// ------------------------
// CDeltaQueryInfoMap
// ------------------------

struct CDeltaQueryInfoMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CDeltaQueryInfoMap);

    inline size_t size() const noexcept {
        return ::CDeltaQueryInfoMap_size(this);
    }

    inline const char* getKey(size_t index) const noexcept {
        return ::CDeltaQueryInfoMap_getKey(this, index);
    }

    inline bool hasValue(size_t index) const noexcept {
        return ::CDeltaQueryInfoMap_getValue(this, index) != nullptr;
    }

    inline const CDeltaQueryInfo& getValue(size_t index) const noexcept {
        return *::CDeltaQueryInfoMap_getValue(this, index);
    }

};

// ------------------------
// CParameters
// ------------------------

struct CParameters {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CParameters);

    inline static const CParameters& getEmptyParameters() noexcept {
        return *::CParameters_getEmptyParameters();
    }

    inline static CParametersPtr newEmptyParameters() {
        CParameters* parameters;
        REPORT_EXCEPTION_IF_NEEDED(::CParameters_newEmptyParameters(&parameters));
        return CParametersPtr{ parameters };
    }

    inline void setString(const char* key, const char* value) {
        REPORT_EXCEPTION_IF_NEEDED(::CParameters_setString(this, key, value));
    }

    inline const char* getString(const char* key, const char* defaultValue) const {
        const char* string;
        REPORT_EXCEPTION_IF_NEEDED(::CParameters_getString(this, key, defaultValue, &string));
        return string;
    }

};

// ------------------------
// CPrefixes
// ------------------------

struct CPrefixes {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CPrefixes);

    inline static const CPrefixes& getEmptyPrefixes() noexcept {
        return *::CPrefixes_getEmptyPrefixes();
    }

    inline static const CPrefixes& getDefaultPrefixes() noexcept {
        return *::CPrefixes_getDefaultPrefixes();
    }

    inline static CPrefixesPtr newEmptyPrefixes() {
        CPrefixes* prefixes;
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_newEmptyPrefixes(&prefixes));
        return CPrefixesPtr{ prefixes };
    }

    inline static CPrefixesPtr newDefaultPrefixes() {
        CPrefixes* prefixes;
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_newDefaultPrefixes(&prefixes));
        return CPrefixesPtr{ prefixes };
    }

    inline void declareStandardPrefixes() {
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_declareStandardPrefixes(this));
    }

    inline CPrefixes_DeclareResult declarePrefix(const char* prefixName, const char* prefixIRI) {
        CPrefixes_DeclareResult declareResult;
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_declarePrefix(this, prefixName, prefixIRI, &declareResult));
        return declareResult;
    }

    inline bool undeclarePrefix(const char* prefixName) {
        bool changed;
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_undeclarePrefix(this, prefixName, &changed));
        return changed;
    }

    inline const char* getPrefix(const char* prefixName) const {
        const char* prefixIRI;
        REPORT_EXCEPTION_IF_NEEDED(::CPrefixes_getPrefix(this, prefixName, &prefixIRI));
        return prefixIRI;
    }

};

// ------------------------
// CPrivilegesMap
// ------------------------

struct CPrivilegesMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CPrivilegesMap);

    inline size_t size() const noexcept {
        return ::CPrivilegesMap_size(this);
    }

    inline const char* getResourceSpecifier(size_t index) const noexcept {
        return ::CPrivilegesMap_getResourceSpecifier(this, index);
    }

    inline CAccessTypeFlags getGrantedAccessTypes(size_t index) const noexcept {
        return ::CPrivilegesMap_getGrantedAccessTypes(this, index);
    }

};

// ------------------------
// CResourceValueTuples
// ------------------------

struct CResourceValueTuples {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CResourceValueTuples);

    inline size_t size() const noexcept {
        return ::CResourceValueTuples_size(this);
    }

    inline size_t arity() const noexcept {
        return ::CResourceValueTuples_arity(this);
    }

    inline void appendResourceLexicalForm(size_t row, size_t column, char* buffer, size_t bufferSize, size_t& lexicalFormSize, CDatatypeID& datatypeID) const {
        REPORT_EXCEPTION_IF_NEEDED(::CResourceValueTuples_appendResourceLexicalForm(this, row, column, buffer, bufferSize, &lexicalFormSize, &datatypeID));
    }

    inline void appendResourceTurtleLiteral(const CPrefixes& prefixes, size_t row, size_t column, char* buffer, size_t bufferSize, size_t& turtleLiteralSize) const {
        REPORT_EXCEPTION_IF_NEEDED(::CResourceValueTuples_appendResourceTurtleLiteral(this, &prefixes, row, column, buffer, bufferSize, &turtleLiteralSize));
    }

};

// ------------------------
// CRDFoxServer
// ------------------------

struct CRDFoxServer {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CRDFoxServer);

    inline static CInstanceState analyzeState(const char* serverDirectoryPath) {
        CInstanceState instanceState;
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_analyzeState(serverDirectoryPath, &instanceState));
        return instanceState;
    }

    inline static void initialize(const char* serverDirectoryPath, const char* persistenceEncryptionKey, const CParameters& persistentServerParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_initialize(serverDirectoryPath, persistenceEncryptionKey, &persistentServerParameters));
    }

    inline static void upgrade(const char* serverDirectoryPath, const char* persistenceEncryptionKey, const COutputStream& outputStream) {
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_upgrade(serverDirectoryPath, persistenceEncryptionKey, &outputStream));
    }

    inline static size_t start(const CParameters& serverParameters) {
        size_t numberOfDataStoresInServer;
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_start(&serverParameters, &numberOfDataStoresInServer));
        return numberOfDataStoresInServer;
    }

    inline static size_t getNumberOfRoles() {
        size_t numberOfRoles;
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_getNumberOfRoles(&numberOfRoles));
        return numberOfRoles;
    }

    inline static void createFirstRole(const char* firstRoleName, const char* password) {
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_createFirstRole(firstRoleName, password));
    }

    inline static CServerConnectionPtr newServerConnection(const char* roleName, const char* password) {
        CServerConnection* serverConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_newServerConnection(roleName, password, &serverConnection));
        return CServerConnectionPtr{ serverConnection };
    }

    inline static CDataStoreConnectionPtr newDataStoreConnection(const char* dataStoreName, const char* roleName, const char* password) {
        CDataStoreConnection* dataStoreConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_newDataStoreConnection(dataStoreName, roleName, password, &dataStoreConnection));
        return CDataStoreConnectionPtr{ dataStoreConnection };
    }

    inline static void stop() {
        REPORT_EXCEPTION_IF_NEEDED(::CRDFoxServer_stop());
    }

};

// ------------------------
// CServerConnection
// ------------------------

struct CServerConnection {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CServerConnection);

    // Connection management

    inline void interrupt() {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_interrupt(this));
    }

    inline CServerConnectionPtr duplicate() {
        CServerConnection* outputServerConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_duplicate(this, &outputServerConnection));
        return CServerConnectionPtr{ outputServerConnection };
    }

    // Server properties

    inline const CParameters& getParameters() {
        const CParameters* parameters;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getParameters(this, &parameters));
        return *parameters;
    }

    inline const char* getRDFoxVersion() {
        const char* rdfoxVersion;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getRDFoxVersion(this, &rdfoxVersion));
        return rdfoxVersion;
    }

    inline const char* getRDFoxGitSHA() {
        const char* rdfoxGitSHA;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getRDFoxGitSHA(this, &rdfoxGitSHA));
        return rdfoxGitSHA;
    }

    inline std::pair<size_t, size_t> getMemoryUse() {
        std::pair<size_t, size_t> memoryUse;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getMemoryUse(this, &memoryUse.first, &memoryUse.second));
        return memoryUse;
    }

    inline void setMaxMemoryUse(size_t maxUsedBytes) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_setMaxMemoryUse(this, maxUsedBytes));
    }

    inline size_t getNumberOfThreads() {
        size_t numberOfThreads;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getNumberOfThreads(this, &numberOfThreads));
        return numberOfThreads;
    }

    inline void setNumberOfThreads(size_t numberOfThreads) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_setNumberOfThreads(this, numberOfThreads));
    }

    inline size_t getServerVersion() {
        size_t serverVersion;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getServerVersion(this, &serverVersion));
        return serverVersion;
    }

    inline bool checkIntegrity(size_t maxErrorsOfSameType, const COutputStream& outputStream) {
        bool errorsDetected;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_checkIntegrity(this, maxErrorsOfSameType, &outputStream, &errorsDetected));
        return errorsDetected;
    }

    inline CComponentInfoPtr getComponentInfo(bool extended) {
        CComponentInfo* componentInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getComponentInfo(this, extended, &componentInfo));
        return CComponentInfoPtr{ componentInfo };
    }

    // Data store management

    inline CDataStoreInfoMapPtr listDataStores() {
        CDataStoreInfoMap* dataStoreInfoMap;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_listDataStores(this, &dataStoreInfoMap));
        return CDataStoreInfoMapPtr{ dataStoreInfoMap };
    }

    inline bool containsDataStore(const char* dataStoreName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_containsDataStore(this, dataStoreName, &result));
        return result;
    }

    inline CDataStoreInfoPtr describeDataStore(const char* dataStoreName) {
        CDataStoreInfo* dataStoreInfo;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_describeDataStore(this, dataStoreName, &dataStoreInfo));
        return CDataStoreInfoPtr{ dataStoreInfo };
    }

    inline void createDataStore(const char* dataStoreName, const CParameters& dataStoreParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_createDataStore(this, dataStoreName, &dataStoreParameters));
    }

    inline void loadDataStoreFromBinaryFormat(const char* dataStoreName, const CInputStream& inputStream) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_loadDataStoreFromBinaryFormat(this, dataStoreName, &inputStream));
    }

    inline void loadDataStoreFromBinaryFormatWithParameters(const char* dataStoreName, const CInputStream& inputStream, const CParameters& dataStoreParameters) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_loadDataStoreFromBinaryFormatWithParameters(this, dataStoreName, &inputStream, &dataStoreParameters));
    }

    inline void saveDataStoreToBinaryFormat(const char* dataStoreName, const COutputStream& outputStream, CBinaryDataStoreFormat format) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_saveDataStoreToBinaryFormat(this, dataStoreName, &outputStream, format));
    }

    inline void saveDataStoreToBinaryFormatWithMonitor(const char* dataStoreName, const COutputStream& outputStream, CBinaryDataStoreFormat format, void* context, CDataStoreOperationStartMonitor dataStoreOperationStartMonitor) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_saveDataStoreToBinaryFormatWithMonitor(this, dataStoreName, &outputStream, format, context, dataStoreOperationStartMonitor));
    }

    inline void deleteDataStore(const char* dataStoreName) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_deleteDataStore(this, dataStoreName));
    }

    inline bool isDataStoreOnline(const char* dataStoreName) {
        bool isOnline;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_isDataStoreOnline(this, dataStoreName, &isOnline));
        return isOnline;
    }

    inline bool bringDataStoreOnline(const char* dataStoreName) {
        bool wasOffline;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_bringDataStoreOnline(this, dataStoreName, &wasOffline));
        return wasOffline;
    }

    inline COfflineRequestResult bringDataStoreOffline(const char* dataStoreName) {
        COfflineRequestResult offlineRequestResult;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_bringDataStoreOffline(this, dataStoreName, &offlineRequestResult));
        return offlineRequestResult;
    }

    inline CDataStoreConnectionPtr newDataStoreConnection(const char* dataStoreName) {
        CDataStoreConnection* dataStoreConnection;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_newDataStoreConnection(this, dataStoreName, &dataStoreConnection));
        return CDataStoreConnectionPtr{ dataStoreConnection };
    }

    // Role management

    inline CSetOfStringsPtr listRoles() {
        CSetOfStrings* setOfRoles;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_listRoles(this, &setOfRoles));
        return CSetOfStringsPtr{ setOfRoles };
    }

    inline void createRole(const char* roleName, CPasswordSpecificationType passwordSpecificationType, const char* password) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_createRole(this, roleName, passwordSpecificationType, password));
    }

    inline void deleteRole(const char* roleName) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_deleteRole(this, roleName));
    }

    inline void changeRolePassword(const char* currentPassword, const char* newPassword) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_changeRolePassword(this, currentPassword, newPassword));
    }

    inline void getRolePasswordHash(const char* roleName, char* buffer, size_t bufferSize, size_t* rolePasswordHashLength) {
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_getRolePasswordHash(this, roleName, buffer, bufferSize, rolePasswordHashLength));
    }

    inline CSetOfStringsPtr listRoleMemberships(const char* roleName) {
        CSetOfStrings* setOfRoleMemberships;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_listRoleMemberships(this, roleName, &setOfRoleMemberships));
        return CSetOfStringsPtr{ setOfRoleMemberships };
    }

    inline CSetOfStringsPtr listRoleMembers(const char* roleName) {
        CSetOfStrings* setOfRoleMembers;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_listRoleMembers(this, roleName, &setOfRoleMembers));
        return CSetOfStringsPtr{ setOfRoleMembers };
    }

    inline bool isMemberOf(const char* roleName, const char* superRoleName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_isMemberOf(this, roleName, superRoleName, &result));
        return result;
    }

    inline bool grantRole(const char* roleName, const char* superRoleName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_grantRole(this, roleName, superRoleName, &result));
        return result;
    }

    inline bool revokeRole(const char* roleName, const char* superRoleName) {
        bool result;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_revokeRole(this, roleName, superRoleName, &result));
        return result;
    }

    inline CPrivilegesMapPtr listPrivileges(const char* roleName) {
        CPrivilegesMap* accessTypeFlagsMap;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_listPrivileges(this, roleName, &accessTypeFlagsMap));
        return CPrivilegesMapPtr{ accessTypeFlagsMap };
    }

    inline bool grantPrivileges(const char* roleName, const char* resourceSpecifier, CAccessTypeFlags accessTypesToGrant) {
        bool granted;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_grantPrivileges(this, roleName, resourceSpecifier, accessTypesToGrant, &granted));
        return granted;
    }

    inline bool revokePrivileges(const char* roleName, const char* resourceSpecifier, CAccessTypeFlags accessTypesToRevoke) {
        bool revoked;
        REPORT_EXCEPTION_IF_NEEDED(::CServerConnection_revokePrivileges(this, roleName, resourceSpecifier, accessTypesToRevoke, &revoked));
        return revoked;
    }

};

// ------------------------
// CSetOfStrings
// ------------------------

struct CSetOfStrings {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CSetOfStrings);

    inline size_t size() const noexcept {
        return ::CSetOfStrings_size(this);
    }

    inline const char* get(size_t index) const noexcept {
        return ::CSetOfStrings_get(this, index);
    }
};

// ------------------------
// CStatisticsInfo
// ------------------------

struct CStatisticsInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CStatisticsInfo);

    inline const char* getName() const noexcept {
        return ::CStatisticsInfo_getName(this);
    }

    inline const CParameters& getParameters() const noexcept {
        return *::CStatisticsInfo_getParameters(this);
    }

};

// ------------------------
// CStatisticsInfoList
// ------------------------

struct CStatisticsInfoList {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CStatisticsInfoList);

    inline size_t size() const noexcept {
        return ::CStatisticsInfoList_size(this);
    }

    inline const CStatisticsInfo& get(size_t index) const noexcept {
        return *::CStatisticsInfoList_get(this, index);
    }

};

// ------------------------
// CTupleTableInfo
// ------------------------

struct CTupleTableInfo {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CTupleTableInfo);

    inline const char* getName() const noexcept {
        return ::CTupleTableInfo_getName(this);
    }

    inline CTupleTableType getType() const noexcept {
        return ::CTupleTableInfo_getType(this);
    }

    inline CTupleTableID getID() const noexcept {
        return ::CTupleTableInfo_getID(this);
    }

    inline size_t getMinArity() const noexcept {
        return ::CTupleTableInfo_getMinArity(this);
    }

    inline size_t getMaxArity() const noexcept {
        return ::CTupleTableInfo_getMaxArity(this);
    }

    inline const CParameters* getParameters() const noexcept {
        return ::CTupleTableInfo_getParameters(this);
    }

};

// ------------------------
// CTupleTableInfoMap
// ------------------------

struct CTupleTableInfoMap {
    DISABLE_CONSTRUCTOR_AND_ASSIGNMENT(CTupleTableInfoMap);

    inline size_t size() const noexcept {
        return ::CTupleTableInfoMap_size(this);
    }

    inline const char* getKey(size_t index) const noexcept {
        return ::CTupleTableInfoMap_getKey(this, index);
    }

    inline bool hasValue(size_t index) const noexcept {
        return ::CTupleTableInfoMap_getValue(this, index) != nullptr;
    }

    inline const CTupleTableInfo& getValue(size_t index) const noexcept {
        return *::CTupleTableInfoMap_getValue(this, index);
    }

};

#undef REPORT_EXCEPTION_IF_NEEDED
#undef DISABLE_CONSTRUCTOR_AND_ASSIGNMENT

#endif

#endif // CRDFOX_H_
